#include "ScalarConverter.hpp"

ScalarConverter::ScalarConverter() {}
ScalarConverter::ScalarConverter(const ScalarConverter& other) { (void)other; }
ScalarConverter& ScalarConverter::operator=(const ScalarConverter& other) { (void)other; return *this; }
ScalarConverter::~ScalarConverter() {}

bool ScalarConverter::isChar(const std::string& literal)
{
	return (literal.length() == 1 && !std::isdigit(literal[0]));
}

bool ScalarConverter::isInt(const std::string& literal)
{
	if (literal.empty())
		return false;
	size_t i = 0;
	if (literal[0] == '+' || literal[0] == '-')
		i++;
	if (i == literal.length())
		return false;
	for (; i < literal.length(); i++)
	{
		if (!std::isdigit(literal[i]))
			return false;
	}
	return true;
}

bool ScalarConverter::isFloat(const std::string& literal)
{
	if (literal.empty())
		return false;
	if (literal == "-inff" || literal == "+inff" || literal == "nanf")
		return true;
	if (literal[literal.length() - 1] != 'f')
		return false;
	std::string withoutF = literal.substr(0, literal.length() - 1);
	if (withoutF.empty())
		return false;
	size_t i = 0;
	if (withoutF[0] == '+' || withoutF[0] == '-')
		i++;
	bool hasDot = false;
	bool hasDigit = false;
	for (; i < withoutF.length(); i++)
	{
		if (withoutF[i] == '.')
		{
			if (hasDot)
				return false;
			hasDot = true;
		}
		else if (std::isdigit(withoutF[i]))
			hasDigit = true;
		else
			return false;
	}
	return hasDot && hasDigit;
}

bool ScalarConverter::isDouble(const std::string& literal)
{
	if (literal.empty())
		return false;
	if (literal == "-inf" || literal == "+inf" || literal == "nan")
		return true;
	size_t i = 0;
	if (literal[0] == '+' || literal[0] == '-')
		i++;
	bool hasDot = false;
	bool hasDigit = false;
	for (; i < literal.length(); i++)
	{
		if (literal[i] == '.')
		{
			if (hasDot)
				return false;
			hasDot = true;
		}
		else if (std::isdigit(literal[i]))
			hasDigit = true;
		else
			return false;
	}
	return hasDot && hasDigit;
}

bool ScalarConverter::isPseudoLiteral(const std::string& literal)
{
	return (literal == "-inff" || literal == "+inff" || literal == "nanf" ||
			literal == "-inf" || literal == "+inf" || literal == "nan");
}

void ScalarConverter::convertFromChar(char c)
{
	std::cout << "char: ";
	if (std::isprint(c))
		std::cout << "'" << c << "'" << std::endl;
	else
		std::cout << "Non displayable" << std::endl;
	std::cout << "int: " << static_cast<int>(c) << std::endl;
	std::cout << "float: " << static_cast<float>(c) << ".0f" << std::endl;
	std::cout << "double: " << static_cast<double>(c) << ".0" << std::endl;
}

void ScalarConverter::convertFromInt(long value)
{
	std::cout << "char: ";
	if (value < 0 || value > 127)
		std::cout << "impossible" << std::endl;
	else if (!std::isprint(static_cast<int>(value)))
		std::cout << "Non displayable" << std::endl;
	else
		std::cout << "'" << static_cast<char>(value) << "'" << std::endl;

	std::cout << "int: ";
	if (value < INT_MIN || value > INT_MAX)
		std::cout << "impossible" << std::endl;
	else
		std::cout << static_cast<int>(value) << std::endl;

	std::cout << "float: " << static_cast<float>(value) << ".0f" << std::endl;
	std::cout << "double: " << static_cast<double>(value) << ".0" << std::endl;
}

void ScalarConverter::convertFromFloat(float value)
{
	std::cout << "char: ";
	if (std::isnan(value) || std::isinf(value) || value < 0 || value > 127)
		std::cout << "impossible" << std::endl;
	else if (!std::isprint(static_cast<int>(value)))
		std::cout << "Non displayable" << std::endl;
	else
		std::cout << "'" << static_cast<char>(value) << "'" << std::endl;

	std::cout << "int: ";
	if (std::isnan(value) || std::isinf(value) || value < static_cast<float>(INT_MIN) || value > static_cast<float>(INT_MAX))
		std::cout << "impossible" << std::endl;
	else
		std::cout << static_cast<int>(value) << std::endl;

	std::cout << "float: " << value;
	if (value == static_cast<int>(value) && !std::isinf(value) && !std::isnan(value))
		std::cout << ".0";
	std::cout << "f" << std::endl;

	std::cout << "double: " << static_cast<double>(value);
	if (value == static_cast<int>(value) && !std::isinf(value) && !std::isnan(value))
		std::cout << ".0";
	std::cout << std::endl;
}

void ScalarConverter::convertFromDouble(double value)
{
	std::cout << "char: ";
	if (std::isnan(value) || std::isinf(value) || value < 0 || value > 127)
		std::cout << "impossible" << std::endl;
	else if (!std::isprint(static_cast<int>(value)))
		std::cout << "Non displayable" << std::endl;
	else
		std::cout << "'" << static_cast<char>(value) << "'" << std::endl;

	std::cout << "int: ";
	if (std::isnan(value) || std::isinf(value) || value < static_cast<double>(INT_MIN) || value > static_cast<double>(INT_MAX))
		std::cout << "impossible" << std::endl;
	else
		std::cout << static_cast<int>(value) << std::endl;

	std::cout << "float: ";
	if (!std::isinf(value) && (value < -FLT_MAX || value > FLT_MAX))
		std::cout << "impossible" << std::endl;
	else
	{
		std::cout << static_cast<float>(value);
		if (value == static_cast<long>(value) && !std::isinf(value) && !std::isnan(value) && value < 1000000 && value > -1000000)
			std::cout << ".0";
		std::cout << "f" << std::endl;
	}

	std::cout << "double: " << value;
	if (value == static_cast<long>(value) && !std::isinf(value) && !std::isnan(value) && value < 1000000 && value > -1000000)
		std::cout << ".0";
	std::cout << std::endl;
}

void ScalarConverter::handlePseudoLiteral(const std::string& literal)
{
	std::cout << "char: impossible" << std::endl;
	std::cout << "int: impossible" << std::endl;
	if (literal == "nanf" || literal == "nan")
	{
		std::cout << "float: nanf" << std::endl;
		std::cout << "double: nan" << std::endl;
	}
	else if (literal == "+inff" || literal == "+inf")
	{
		std::cout << "float: +inff" << std::endl;
		std::cout << "double: +inf" << std::endl;
	}
	else
	{
		std::cout << "float: -inff" << std::endl;
		std::cout << "double: -inf" << std::endl;
	}
}

void ScalarConverter::convert(const std::string& literal)
{
	if (literal.empty())
	{
		std::cout << "Error: empty literal" << std::endl;
		return;
	}

	if (isPseudoLiteral(literal))
	{
		handlePseudoLiteral(literal);
		return;
	}

	if (isChar(literal))
	{
		convertFromChar(literal[0]);
		return;
	}

	if (isInt(literal))
	{
		errno = 0;
		char* endptr;
		long value = std::strtol(literal.c_str(), &endptr, 10);
		if (errno == ERANGE)
		{
			std::cout << "char: impossible" << std::endl;
			std::cout << "int: impossible" << std::endl;
			std::cout << "float: impossible" << std::endl;
			std::cout << "double: impossible" << std::endl;
			return;
		}
		convertFromInt(value);
		return;
	}

	if (isFloat(literal))
	{
		std::string withoutF = literal.substr(0, literal.length() - 1);
		errno = 0;
		char* endptr;
		float value = std::strtof(withoutF.c_str(), &endptr);
		if (errno == ERANGE && !std::isinf(value))
		{
			std::cout << "char: impossible" << std::endl;
			std::cout << "int: impossible" << std::endl;
			std::cout << "float: impossible" << std::endl;
			std::cout << "double: impossible" << std::endl;
			return;
		}
		convertFromFloat(value);
		return;
	}

	if (isDouble(literal))
	{
		errno = 0;
		char* endptr;
		double value = std::strtod(literal.c_str(), &endptr);
		if (errno == ERANGE && !std::isinf(value))
		{
			std::cout << "char: impossible" << std::endl;
			std::cout << "int: impossible" << std::endl;
			std::cout << "float: impossible" << std::endl;
			std::cout << "double: impossible" << std::endl;
			return;
		}
		convertFromDouble(value);
		return;
	}

	std::cout << "Error: invalid literal" << std::endl;
}
