#include "PmergeMe.hpp"

PmergeMe::PmergeMe() : _vectorTime(0), _dequeTime(0) {}

PmergeMe::~PmergeMe() {}

PmergeMe::PmergeMe(const PmergeMe& other) 
    : _vec(other._vec), _deq(other._deq), 
      _vectorTime(other._vectorTime), _dequeTime(other._dequeTime) {}

PmergeMe& PmergeMe::operator=(const PmergeMe& other) {
    if (this != &other) {
        _vec = other._vec;
        _deq = other._deq;
        _vectorTime = other._vectorTime;
        _dequeTime = other._dequeTime;
    }
    return *this;
}

size_t PmergeMe::jacobsthal(size_t n) const {
    if (n == 0) return 0;
    if (n == 1) return 1;
    
    size_t prev2 = 0;
    size_t prev1 = 1;
    size_t current = 0;
    
    for (size_t i = 2; i <= n; i++) {
        current = prev1 + 2 * prev2;
        prev2 = prev1;
        prev1 = current;
    }
    return current;
}

void PmergeMe::parseInput(int argc, char** argv) {
    for (int i = 1; i < argc; i++) {
        std::string arg(argv[i]);
        
        if (arg.empty())
            throw std::runtime_error("Error");
        
        for (size_t j = 0; j < arg.length(); j++) {
            if (!std::isdigit(arg[j]) && !(j == 0 && arg[j] == '+'))
                throw std::runtime_error("Error");
        }
        
        long num = std::atol(arg.c_str());
        if (num < 0 || num > 2147483647)
            throw std::runtime_error("Error");
        
        _vec.push_back(static_cast<int>(num));
        _deq.push_back(static_cast<int>(num));
    }
    
    if (_vec.empty())
        throw std::runtime_error("Error");
}

void PmergeMe::insertWithBinarySearchVector(std::vector<int>& sorted, int value, size_t maxPos) {
    size_t left = 0;
    size_t right = maxPos;
    
    while (left < right) {
        size_t mid = left + (right - left) / 2;
        if (sorted[mid] > value)
            right = mid;
        else
            left = mid + 1;
    }
    sorted.insert(sorted.begin() + static_cast<long>(left), value);
}

void PmergeMe::mergeInsertSortVector(std::vector<int>& arr) {
    size_t n = arr.size();
    if (n <= 1)
        return;
    
    if (n == 2) {
        if (arr[0] > arr[1])
            std::swap(arr[0], arr[1]);
        return;
    }
    
    std::vector<std::pair<int, int> > pairs;
    int straggler = -1;
    bool hasStraggler = (n % 2 == 1);
    
    if (hasStraggler)
        straggler = arr[n - 1];
    
    for (size_t i = 0; i + 1 < n; i += 2) {
        int larger = arr[i] > arr[i + 1] ? arr[i] : arr[i + 1];
        int smaller = arr[i] > arr[i + 1] ? arr[i + 1] : arr[i];
        pairs.push_back(std::make_pair(larger, smaller));
    }
    
    std::vector<int> largerElements;
    for (size_t i = 0; i < pairs.size(); i++)
        largerElements.push_back(pairs[i].first);
    
    mergeInsertSortVector(largerElements);
    
    std::vector<std::pair<int, int> > sortedPairs;
    std::vector<bool> used(pairs.size(), false);
    for (size_t i = 0; i < largerElements.size(); i++) {
        for (size_t j = 0; j < pairs.size(); j++) {
            if (!used[j] && pairs[j].first == largerElements[i]) {
                sortedPairs.push_back(pairs[j]);
                used[j] = true;
                break;
            }
        }
    }
    pairs = sortedPairs;
    
    std::vector<int> mainChain;
    std::vector<int> pend;
    
    mainChain.push_back(pairs[0].second);
    mainChain.push_back(pairs[0].first);
    
    for (size_t i = 1; i < pairs.size(); i++) {
        mainChain.push_back(pairs[i].first);
        pend.push_back(pairs[i].second);
    }
    
    size_t pendSize = pend.size();
    
    if (pendSize > 0) {
        std::vector<size_t> insertOrder;
        
        size_t k = 3;
        size_t prev = 1;
        while (insertOrder.size() < pendSize) {
            size_t curr = jacobsthal(k);
            size_t pos = curr - 1;
            if (pos >= pendSize)
                pos = pendSize - 1;
            
            while (pos >= prev - 1 && insertOrder.size() < pendSize) {
                bool found = false;
                for (size_t i = 0; i < insertOrder.size(); i++) {
                    if (insertOrder[i] == pos) {
                        found = true;
                        break;
                    }
                }
                if (!found)
                    insertOrder.push_back(pos);
                if (pos == 0) break;
                pos--;
            }
            prev = curr;
            k++;
        }
        
        for (size_t i = 0; i < insertOrder.size(); i++) {
            size_t pendIdx = insertOrder[i];
            size_t bound = pendIdx + 2 + i;
            if (bound > mainChain.size())
                bound = mainChain.size();
            insertWithBinarySearchVector(mainChain, pend[pendIdx], bound);
        }
    }
    
    if (hasStraggler)
        insertWithBinarySearchVector(mainChain, straggler, mainChain.size());
    
    arr = mainChain;
}

void PmergeMe::insertWithBinarySearchDeque(std::deque<int>& sorted, int value, size_t maxPos) {
    size_t left = 0;
    size_t right = maxPos;
    
    while (left < right) {
        size_t mid = left + (right - left) / 2;
        if (sorted[mid] > value)
            right = mid;
        else
            left = mid + 1;
    }
    sorted.insert(sorted.begin() + static_cast<long>(left), value);
}

void PmergeMe::mergeInsertSortDeque(std::deque<int>& arr) {
    size_t n = arr.size();
    if (n <= 1)
        return;
    
    if (n == 2) {
        if (arr[0] > arr[1])
            std::swap(arr[0], arr[1]);
        return;
    }
    
    std::deque<std::pair<int, int> > pairs;
    int straggler = -1;
    bool hasStraggler = (n % 2 == 1);
    
    if (hasStraggler)
        straggler = arr[n - 1];
    
    for (size_t i = 0; i + 1 < n; i += 2) {
        int larger = arr[i] > arr[i + 1] ? arr[i] : arr[i + 1];
        int smaller = arr[i] > arr[i + 1] ? arr[i + 1] : arr[i];
        pairs.push_back(std::make_pair(larger, smaller));
    }
    
    std::deque<int> largerElements;
    for (size_t i = 0; i < pairs.size(); i++)
        largerElements.push_back(pairs[i].first);
    
    mergeInsertSortDeque(largerElements);
    
    std::deque<std::pair<int, int> > sortedPairs;
    std::deque<bool> used(pairs.size(), false);
    for (size_t i = 0; i < largerElements.size(); i++) {
        for (size_t j = 0; j < pairs.size(); j++) {
            if (!used[j] && pairs[j].first == largerElements[i]) {
                sortedPairs.push_back(pairs[j]);
                used[j] = true;
                break;
            }
        }
    }
    pairs = sortedPairs;
    
    std::deque<int> mainChain;
    std::deque<int> pend;
    
    mainChain.push_back(pairs[0].second);
    mainChain.push_back(pairs[0].first);
    
    for (size_t i = 1; i < pairs.size(); i++) {
        mainChain.push_back(pairs[i].first);
        pend.push_back(pairs[i].second);
    }
    
    size_t pendSize = pend.size();
    
    if (pendSize > 0) {
        std::deque<size_t> insertOrder;
        
        size_t k = 3;
        size_t prev = 1;
        while (insertOrder.size() < pendSize) {
            size_t curr = jacobsthal(k);
            size_t pos = curr - 1;
            if (pos >= pendSize)
                pos = pendSize - 1;
            
            while (pos >= prev - 1 && insertOrder.size() < pendSize) {
                bool found = false;
                for (size_t i = 0; i < insertOrder.size(); i++) {
                    if (insertOrder[i] == pos) {
                        found = true;
                        break;
                    }
                }
                if (!found)
                    insertOrder.push_back(pos);
                if (pos == 0) break;
                pos--;
            }
            prev = curr;
            k++;
        }
        
        for (size_t i = 0; i < insertOrder.size(); i++) {
            size_t pendIdx = insertOrder[i];
            size_t bound = pendIdx + 2 + i;
            if (bound > mainChain.size())
                bound = mainChain.size();
            insertWithBinarySearchDeque(mainChain, pend[pendIdx], bound);
        }
    }
    
    if (hasStraggler)
        insertWithBinarySearchDeque(mainChain, straggler, mainChain.size());
    
    arr = mainChain;
}

void PmergeMe::sort() {
    struct timeval start, end;
    
    std::vector<int> vecCopy = _vec;
    gettimeofday(&start, NULL);
    mergeInsertSortVector(vecCopy);
    gettimeofday(&end, NULL);
    _vec = vecCopy;
    _vectorTime = (end.tv_sec - start.tv_sec) * 1000000.0 + (end.tv_usec - start.tv_usec);
    
    std::deque<int> deqCopy = _deq;
    gettimeofday(&start, NULL);
    mergeInsertSortDeque(deqCopy);
    gettimeofday(&end, NULL);
    _deq = deqCopy;
    _dequeTime = (end.tv_sec - start.tv_sec) * 1000000.0 + (end.tv_usec - start.tv_usec);
}

void PmergeMe::displayBefore() const {
    std::cout << "Before:";
    for (size_t i = 0; i < _vec.size() && i < 10; i++)
        std::cout << " " << _vec[i];
    if (_vec.size() > 10)
        std::cout << " [...]";
    std::cout << std::endl;
}

void PmergeMe::displayAfter() const {
    std::cout << "After:";
    for (size_t i = 0; i < _vec.size() && i < 10; i++)
        std::cout << " " << _vec[i];
    if (_vec.size() > 10)
        std::cout << " [...]";
    std::cout << std::endl;
}

double PmergeMe::getVectorTime() const {
    return _vectorTime;
}

double PmergeMe::getDequeTime() const {
    return _dequeTime;
}
